/****************************************************************************
**
** Copyright (C) 2021 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the tools applications of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:GPL-EXCEPT$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3 as published by the Free Software
** Foundation with exceptions as appearing in the file LICENSE.GPL3-EXCEPT
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qmltccodewriter.h"

#include <QtCore/qfileinfo.h>
#include <QtCore/qstringbuilder.h>

#include <utility>

QT_BEGIN_NAMESPACE

static QString urlToMacro(const QString &url)
{
    QFileInfo fi(url);
    return u"Q_QMLTC_" + fi.baseName().toUpper();
}

static QString getFunctionCategory(const QmltcMethodBase &method)
{
    QString category;
    switch (method.access) {
    case QQmlJSMetaMethod::Private:
        category = u"private"_qs;
        break;
    case QQmlJSMetaMethod::Protected:
        category = u"protected"_qs;
        break;
    case QQmlJSMetaMethod::Public:
        category = u"public"_qs;
        break;
    }
    return category;
}

static QString getFunctionCategory(const QmltcMethod &method)
{
    QString category = getFunctionCategory(static_cast<const QmltcMethodBase &>(method));
    switch (method.type) {
    case QQmlJSMetaMethod::Signal:
        category = u"Q_SIGNALS"_qs;
        break;
    case QQmlJSMetaMethod::Slot:
        category += u" Q_SLOTS"_qs;
        break;
    case QQmlJSMetaMethod::Method:
        break;
    }
    return category;
}

static std::pair<QString, QString> functionSignatures(const QmltcMethodBase &method)
{
    const QString name = method.name;
    const QList<QmltcVariable> &parameterList = method.parameterList;

    QStringList headerParamList;
    QStringList cppParamList;
    for (const QmltcVariable &variable : parameterList) {
        const QString commonPart = variable.cppType + u" " + variable.name;
        cppParamList << commonPart;
        headerParamList << commonPart;
        if (!variable.defaultValue.isEmpty())
            headerParamList.back() += u" = " + variable.defaultValue;
    }

    const QString headerSignature = name + u"(" + headerParamList.join(u", "_qs) + u")";
    const QString cppSignature = name + u"(" + cppParamList.join(u", "_qs) + u")";
    return { headerSignature, cppSignature };
}

void QmltcCodeWriter::writeGlobalHeader(QmltcOutputWrapper &code, const QString &sourcePath,
                                        const QString &hPath, const QString &cppPath,
                                        const QString &outNamespace,
                                        const QSet<QString> &requiredCppIncludes)
{
    Q_UNUSED(cppPath);
    const QString preamble = u"// This code is auto-generated by the qmltc tool from the file '"
            + sourcePath + u"'\n// WARNING! All changes made in this file will be lost!\n";
    code.rawAppendToHeader(preamble);
    code.rawAppendToCpp(preamble);
    code.rawAppendToHeader(
            u"// NOTE: This generated API is to be considered implementation detail.");
    code.rawAppendToHeader(
            u"//       It may change from version to version and should not be relied upon.");

    const QString headerMacro = urlToMacro(sourcePath);
    code.rawAppendToHeader(u"#ifndef %1_H"_qs.arg(headerMacro));
    code.rawAppendToHeader(u"#define %1_H"_qs.arg(headerMacro));

    code.rawAppendToHeader(u"#include <QtCore/qproperty.h>");
    code.rawAppendToHeader(u"#include <QtCore/qobject.h>");
    code.rawAppendToHeader(u"#include <QtCore/qcoreapplication.h>");
    code.rawAppendToHeader(u"#include <QtQml/qqmlengine.h>");
    code.rawAppendToHeader(u"#include <QtCore/qurl.h>"); // used in engine execution
    code.rawAppendToHeader(u"#include <QtQml/qqml.h>"); // used for attached properties

    code.rawAppendToHeader(u"#include <private/qqmlengine_p.h>"); // executeRuntimeFunction(), etc.

    code.rawAppendToHeader(u"#include <QtQml/qqmllist.h>"); // QQmlListProperty

    // include custom C++ includes required by used types
    code.rawAppendToHeader(u"// BEGIN(custom_cpp_includes)");
    for (const auto &requiredInclude : requiredCppIncludes)
        code.rawAppendToHeader(u"#include \"" + requiredInclude + u"\"");
    code.rawAppendToHeader(u"// END(custom_cpp_includes)");
    code.rawAppendToHeader(u"// qmltc support library:");
    code.rawAppendToHeader(u"#include <private/qqmltcobjectcreationhelper_p.h>");

    code.rawAppendToCpp(u"#include \"" + hPath + u"\""); // include own .h file

    code.rawAppendToCpp(u""); // blank line
    code.rawAppendToCpp(u"#include <private/qobject_p.h>"); // NB: for private properties
    code.rawAppendToCpp(u"#include <private/qqmlglobal_p.h>"); // QQml_setParent_noEvent()

    code.rawAppendToCpp(u""); // blank line
    code.rawAppendToCpp(u"QT_USE_NAMESPACE // avoid issues with QT_NAMESPACE");
    if (!outNamespace.isEmpty()) {
        code.rawAppendToHeader(u""); // blank line
        code.rawAppendToHeader(u"namespace %1 {"_qs.arg(outNamespace));
        code.rawAppendToCpp(u""); // blank line
        code.rawAppendToCpp(u"namespace %1 {"_qs.arg(outNamespace));
    }
}

void QmltcCodeWriter::writeGlobalFooter(QmltcOutputWrapper &code, const QString &sourcePath,
                                        const QString &outNamespace)
{
    if (!outNamespace.isEmpty()) {
        code.rawAppendToCpp(u"} // namespace %1"_qs.arg(outNamespace));
        code.rawAppendToCpp(u""); // blank line
        code.rawAppendToHeader(u"} // namespace %1"_qs.arg(outNamespace));
        code.rawAppendToHeader(u""); // blank line
    }

    code.rawAppendToHeader(u"#endif // %1_H"_qs.arg(urlToMacro(sourcePath)));
    code.rawAppendToHeader(u""); // blank line
}

static void writeToFile(const QString &path, const QByteArray &data)
{
    // When not using dependency files, changing a single qml invalidates all
    // qml files and would force the recompilation of everything. To avoid that,
    // we check if the data is equal to the existing file, if yes, don't touch
    // it so the build system will not recompile unnecessary things.
    //
    // If the build system use dependency file, we should anyway touch the file
    // so qmltc is not re-run
    QFileInfo fi(path);
    if (fi.exists() && fi.size() == data.size()) {
        QFile oldFile(path);
        oldFile.open(QIODevice::ReadOnly);
        if (oldFile.readAll() == data)
            return;
    }
    QFile file(path);
    file.open(QIODevice::WriteOnly);
    file.write(data);
}

void QmltcCodeWriter::write(QmltcOutputWrapper &code, const QmltcProgram &program)
{
    writeGlobalHeader(code, program.url, program.hPath, program.cppPath, program.outNamespace,
                      program.includes);
    // TODO: keep the "NOT IMPLEMENTED" as long as we don't actually compile
    // useful code
    code.rawAppendToHeader(u"/* QMLTC: NOT IMPLEMENTED */");
    code.rawAppendToCpp(u"/* QMLTC: NOT IMPLEMENTED */");

    // url method comes first
    writeUrl(code, program.urlMethod);

    // forward declare all the types first
    for (const QmltcType &type : qAsConst(program.compiledTypes))
        code.rawAppendToHeader(u"class " + type.cppType + u";");
    // write all the types and their content
    for (const QmltcType &type : qAsConst(program.compiledTypes))
        write(code, type);
    writeGlobalFooter(code, program.url, program.outNamespace);

    writeToFile(program.hPath, code.code().header.toUtf8());
    writeToFile(program.cppPath, code.code().cpp.toUtf8());
}

void QmltcCodeWriter::write(QmltcOutputWrapper &code, const QmltcType &type)
{
    const auto constructClassString = [&]() {
        QString str = u"class " + type.cppType;
        if (!type.baseClasses.isEmpty())
            str += u" : public " + type.baseClasses.join(u", public "_qs);
        return str;
    };

    code.rawAppendToHeader(u""); // blank line
    code.rawAppendToCpp(u""); // blank line

    code.rawAppendToHeader(constructClassString());
    code.rawAppendToHeader(u"{");
    for (const QString &mocLine : qAsConst(type.mocCode))
        code.rawAppendToHeader(mocLine, 1);

    for (const QString &otherLine : qAsConst(type.otherCode))
        code.rawAppendToHeader(otherLine, 1);

    QmltcOutputWrapper::MemberNameScope typeScope(&code, type.cppType);
    Q_UNUSED(typeScope);
    {
        QmltcOutputWrapper::HeaderIndentationScope headerIndent(&code);
        Q_UNUSED(headerIndent);

        // special member functions
        code.rawAppendToHeader(u"protected:", -1);
        write(code, type.basicCtor);
        write(code, type.init);
        write(code, type.finalize);
        // NB: externalCtor might not be public when the type is QML singleton
        code.rawAppendToHeader(getFunctionCategory(type.fullCtor) + u":", -1);
        write(code, type.fullCtor);

        // enums
        if (!type.enums.isEmpty()) {
            code.rawAppendToHeader(u""); // blank line
            code.rawAppendToHeader(u"public:", -1);
        }
        for (const auto &enumeration : qAsConst(type.enums))
            write(code, enumeration);

        // child types
        if (!type.children.isEmpty())
            code.rawAppendToHeader(u""); // blank line
        for (const auto &child : qAsConst(type.children))
            write(code, child);

        // functions (special case due to functions/signals/slots, etc.)
        QHash<QString, QList<const QmltcMethod *>> functionsByCategory;
        for (const auto &function : qAsConst(type.functions))
            functionsByCategory[getFunctionCategory(function)].append(&function);

        if (!functionsByCategory.isEmpty())
            code.rawAppendToHeader(u""); // blank line
        for (auto it = functionsByCategory.cbegin(); it != functionsByCategory.cend(); ++it) {
            code.rawAppendToHeader(it.key() + u":", -1);
            for (const QmltcMethod *function : qAsConst(it.value()))
                write(code, *function);
        }

        // variables and properties
        if (!type.variables.isEmpty() || !type.properties.isEmpty()) {
            code.rawAppendToHeader(u""); // blank line
            code.rawAppendToHeader(u"protected:", -1);
        }
        for (const auto &property : qAsConst(type.properties))
            write(code, property);
        for (const auto &variable : qAsConst(type.variables))
            write(code, variable);
    }

    if (type.typeCount) {
        // we know that typeCount variable is very special
        code.rawAppendToHeader(u""); // blank line
        code.rawAppendToHeader(u"protected:");
        Q_ASSERT(!type.typeCount->defaultValue.isEmpty());
        code.rawAppendToHeader(u"constexpr static %1 %2 = %3;"_qs.arg(type.typeCount->cppType,
                                                                      type.typeCount->name,
                                                                      type.typeCount->defaultValue),
                               1);
    }

    code.rawAppendToHeader(u"};");
}

void QmltcCodeWriter::write(QmltcOutputWrapper &code, const QmltcEnum &enumeration)
{
    code.rawAppendToHeader(u"enum " + enumeration.cppType + u" {");
    for (qsizetype i = 0; i < enumeration.keys.size(); ++i) {
        QString str;
        if (enumeration.values.isEmpty()) {
            str += enumeration.keys.at(i) + u",";
        } else {
            str += enumeration.keys.at(i) + u" = " + enumeration.values.at(i) + u",";
        }
        code.rawAppendToHeader(str, 1);
    }
    code.rawAppendToHeader(u"};");
    code.rawAppendToHeader(enumeration.ownMocLine);
}

void QmltcCodeWriter::write(QmltcOutputWrapper &code, const QmltcMethod &method)
{
    const auto [hSignature, cppSignature] = functionSignatures(method);
    // Note: augment return type with preambles in declaration
    QString prefix = method.declarationPrefixes.join(u' ');
    if (!prefix.isEmpty())
        prefix.append(u' ');
    code.rawAppendToHeader(prefix + method.returnType + u" " + hSignature + u";");

    // do not generate method implementation if it is a signal
    const auto methodType = method.type;
    if (methodType != QQmlJSMetaMethod::Signal) {
        code.rawAppendToCpp(u""); // blank line
        code.rawAppendToCpp(method.returnType);
        code.rawAppendSignatureToCpp(cppSignature);
        code.rawAppendToCpp(u"{");
        {
            QmltcOutputWrapper::CppIndentationScope cppIndent(&code);
            Q_UNUSED(cppIndent);
            for (const QString &line : qAsConst(method.body))
                code.rawAppendToCpp(line);
        }
        code.rawAppendToCpp(u"}");
    }
}

void QmltcCodeWriter::write(QmltcOutputWrapper &code, const QmltcCtor &ctor)
{
    const auto [hSignature, cppSignature] = functionSignatures(ctor);
    QString prefix = ctor.declarationPrefixes.join(u' ');
    if (!prefix.isEmpty())
        prefix.append(u' ');
    code.rawAppendToHeader(prefix + hSignature + u";");

    code.rawAppendToCpp(u""); // blank line
    code.rawAppendSignatureToCpp(cppSignature);
    if (!ctor.initializerList.isEmpty()) {
        code.rawAppendToCpp(u":", 1);
        // double \n to make separate initializer list lines stand out more
        code.rawAppendToCpp(
                ctor.initializerList.join(u",\n\n" + u"    "_qs.repeated(code.cppIndent + 1)), 1);
    }
    code.rawAppendToCpp(u"{");
    {
        QmltcOutputWrapper::CppIndentationScope cppIndent(&code);
        Q_UNUSED(cppIndent);
        for (const QString &line : qAsConst(ctor.body))
            code.rawAppendToCpp(line);
    }
    code.rawAppendToCpp(u"}");
}

void QmltcCodeWriter::write(QmltcOutputWrapper &code, const QmltcVariable &var)
{
    const QString optionalPart = var.defaultValue.isEmpty() ? u""_qs : u" = " + var.defaultValue;
    code.rawAppendToHeader(var.cppType + u" " + var.name + optionalPart + u";");
}

void QmltcCodeWriter::write(QmltcOutputWrapper &code, const QmltcProperty &prop)
{
    Q_ASSERT(prop.defaultValue.isEmpty()); // we don't support it yet
    code.rawAppendToHeader(u"Q_OBJECT_BINDABLE_PROPERTY(%1, %2, %3, &%1::%4)"_qs.arg(
            prop.containingClass, prop.cppType, prop.name, prop.signalName));
}

void QmltcCodeWriter::writeUrl(QmltcOutputWrapper &code, const QmltcMethod &urlMethod)
{
    // unlike ordinary methods, url function only exists in .cpp
    Q_ASSERT(!urlMethod.returnType.isEmpty());
    const auto [hSignature, _] = functionSignatures(urlMethod);
    Q_UNUSED(_);
    // Note: augment return type with preambles in declaration
    QString prefix = urlMethod.declarationPrefixes.join(u' ');
    if (!prefix.isEmpty())
        prefix.append(u' ');
    code.rawAppendToCpp(prefix + urlMethod.returnType + u" " + hSignature);
    code.rawAppendToCpp(u"{");
    {
        QmltcOutputWrapper::CppIndentationScope cppIndent(&code);
        Q_UNUSED(cppIndent);
        for (const QString &line : qAsConst(urlMethod.body))
            code.rawAppendToCpp(line);
    }
    code.rawAppendToCpp(u"}");
}

QT_END_NAMESPACE
